﻿using System.Runtime.CompilerServices;

namespace PokerAIRL_Sharp;

public class Card
{
    public CardSuit Suit { get; }
    public int Rank { get; }

    public Card(CardSuit suit, int rank)
    {
        Suit = suit;
        Rank = rank;
    }

    public Card(int suit, int rank)
    {
        Suit = (CardSuit)suit;
        Rank = rank;
    }

    public override string ToString() => $"{Rank} {Suit}";

    public string ToNoSuitString() => string.Format("{0}{1}", Rank, Suit.ToString()[..1]);

    public override int GetHashCode()
    {
        return HashCode.Combine(Suit, Rank);
    }

    public override bool Equals(object? obj)
    {
        if (obj is Card other)
        {
            return Suit == other.Suit && Rank == other.Rank;
        }

        return false;
    }

    public static bool operator ==(Card? a, Card? b)
    {
        if ((a is null) != (b is null))
        {
            return false;
        }

        if ((a is null) && (b is null))
        {
            return true;
        }

        return a!.Equals(b);
    }

    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    public static bool operator !=(Card? a, Card? b)
    {
        return !(a == b);
    }

    public static Card[] GenerateCards(int maxSuit)
    {
        return Enumerable
            .Range(0, Math.Min(maxSuit + 1, 3))
            .SelectMany(suit => Enumerable.Range(2, 13).Reverse().Select(rank => new Card(suit, rank)))
            .ToArray();
    }

    public static Card FromNumber(int number) => new Card(number / 13, number % 13 + 2);
}
