﻿using System.Runtime.CompilerServices;

namespace PokerAIRL_Sharp;

public class Hand
{
    public IReadOnlyList<Card> Cards { get; }
    protected readonly int _hashCode;
    public HandRank HandRank { get; }

    public Hand(IEnumerable<Card> cards)
    {
        Cards = SanitizeHand(cards);
        _hashCode = Cards
            .Select(t => t.GetHashCode())
            .Aggregate(HashCode.Combine);
        HandRank = (Cards.Count >= 7) ? CalculateHandRank() : HandRank.Empty;
    }

    public static Card[] SanitizeHand(IEnumerable<Card> cards)
    {
        return cards
            .OrderByDescending(t => t.Rank)
            .ThenBy(t => t.Suit)
            .ToArray();
    }

    public bool HasCard(Card card) => Cards.Contains(card);

    #region Стандартные методы

    public override int GetHashCode()
    {
        return _hashCode;
    }

    public override bool Equals(object? obj)
    {
        if (obj is Hand other)
        {
            return Cards.SequenceEqual(other.Cards);
        }

        return false;
    }

    public static bool operator ==(Hand? a, Hand? b)
    {
        if ((a is null) != (b is null))
        {
            return false;
        }

        if ((a is null) && (b is null))
        {
            return true;
        }

        return a!.Equals(b);
    }

    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    public static bool operator !=(Hand? a, Hand? b)
    {
        return !(a == b);
    }

    public static Hand operator +(Hand a, Hand b)
    {
        return new Hand(a.Cards.Concat(b.Cards));
    }

    public static Hand operator +(Hand a, IEnumerable<Card> cards)
    {
        return new Hand(a.Cards.Concat(cards));
    }

    public override string ToString()
    {
        var value = Cards
            .Select(t => t.ToNoSuitString())
            .Aggregate((a, b) => a + " " + b);

        return $"{value} [{HandRank}]";
    }

    #endregion

    #region Hand Rank

    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    public HandRank CalculateHandRank()
    {
        return CalculateHandRank(this);
    }

    public static HandRank CalculateHandRank(Hand hand)
    {
        #region Straight Flush

        {
            var a = hand
                .Cards
                .Select(t => t.Suit)
                .GroupBy(t => t)
                .Select(t => (suit: t.Key, count: t.Count()))
                .MaxBy(t => t.count);
            if (a.count >= 5)
            {
                var ranks = hand
                    .Cards
                    .Where(x => x.Suit == a.suit)
                    .Select(t => t.Rank)
                    .OrderByDescending(t => t)
                    .ToList();
                if (ranks[0] == 14)
                {
                    // Туз может служить единицей
                    ranks.Add(1);
                }

                var u = false;
                int j;
                for (j = 0; j < ranks.Count - 4; j++)
                {
                    u = true;
                    for (int i = 0; i < 4; i++)
                    {
                        if (ranks[j + i] != ranks[j + i + 1] + 1)
                        {
                            u = false;
                            break;
                        }
                    }

                    if (u) break;
                }

                if (u)
                {
                    return new HandRank(HandRank.HandRankType.StraightFlush, ranks[j]);
                }
            }
        }

        #endregion

        #region Four of a kind

        {
            var a = hand
                .Cards
                .Select(t => t.Rank)
                .GroupBy(t => t)
                .Select(t => (rank: t.Key, count: t.Count()))
                .MaxBy(t => t.count);
            if (a.count >= 4)
            {
                var type2 = hand
                    .Cards
                    .Where(x => x.Rank != a.rank)
                    .Max(t => t.Rank);
                return new HandRank(HandRank.HandRankType.FourOfAKind, a.rank, type2);
            }
        }

        #endregion

        #region Full house

        {
            var a = hand
                .Cards
                .Select(t => t.Rank)
                .GroupBy(t => t)
                .Select(t => (rank: t.Key, count: t.Count()))
                .OrderByDescending(t => t.count)
                .ThenByDescending(t => t.rank)
                .First();
            if (a.count >= 3)
            {
                var type2 = hand
                    .Cards
                    .Where(x => x.Rank != a.rank)
                    .GroupBy(t => t.Rank)
                    .MaxBy(t => t.Count())!;
                if (type2.Count() >= 2)
                {
                    return new HandRank(HandRank.HandRankType.FullHouse, a.rank, type2.Key);
                }
            }
        }

        #endregion

        #region Flush

        {
            var a = hand
                .Cards
                .Select(t => t.Suit)
                .GroupBy(t => t)
                .Select(t => (suit: t.Key, count: t.Count()))
                .MaxBy(t => t.count);
            if (a.count >= 5)
            {
                var ranks = hand
                    .Cards
                    .Where(x => x.Suit == a.suit)
                    .Select(t => t.Rank)
                    .OrderByDescending(t => t)
                    .ToArray();
                return new HandRank(HandRank.HandRankType.Flush, ranks[0], ranks[1], ranks[2], ranks[3], ranks[4]);
            }
        }

        #endregion

        #region Straight

        {
            var has = Enumerable.Repeat(false, 15).ToArray();
            foreach (var rank in hand.Cards.Select(t => t.Rank).Distinct())
            {
                has[rank] = true;
            }

            if (has[14])
            {
                // Туз может служить единицей
                has[1] = true;
            }

            for (var rank = 14; rank >= 5; rank--)
            {
                var u = true;
                for (var i = rank; i >= rank - (5 - 1); i--)
                {
                    if (!has[i])
                    {
                        u = false;
                        break;
                    }
                }

                if (u)
                {
                    return new HandRank(HandRank.HandRankType.Straight, rank);
                }
            }
        }

        #endregion

        #region Three of a kind

        {
            var a = hand
                .Cards
                .Select(t => t.Rank)
                .GroupBy(t => t)
                .Select(t => (rank: t.Key, count: t.Count()))
                .OrderByDescending(t => t.count)
                .ThenByDescending(t => t.rank)
                .First();
            if (a.count >= 3)
            {
                var ranks = hand
                    .Cards
                    .Where(x => x.Rank != a.rank)
                    .Select(t => t.Rank)
                    .OrderByDescending(t => t)
                    .Take(2)
                    .ToArray();
                return new HandRank(HandRank.HandRankType.ThreeOfAKind, a.rank, ranks[0], ranks[1]);
            }
        }

        #endregion

        #region Two pair

        {
            var a = hand
                .Cards
                .Select(t => t.Rank)
                .GroupBy(t => t)
                .Select(t => (rank: t.Key, count: t.Count()))
                .OrderByDescending(t => t.count)
                .ThenByDescending(t => t.rank)
                .ToDictionary(t => t.rank, t => t.count);
            var keys = a.Keys.ToArray();
            if ((keys.Length >= 3) && (a[keys[1]] == 2))
            {
                return new HandRank(HandRank.HandRankType.TwoPair, keys[0], keys[1], keys[2]);
            }
        }

        #endregion

        #region One pair

        {
            var a = hand
                .Cards
                .Select(t => t.Rank)
                .GroupBy(t => t)
                .Select(t => (rank: t.Key, count: t.Count()))
                .Where(x => x.count == 2)
                .OrderByDescending(t => t.rank)
                .FirstOrDefault();
            if (a != default)
            {
                var rank1 = hand
                    .Cards
                    .Select(t => t.Rank)
                    .Where(x => x != a.rank)
                    .Take(3)
                    .ToArray();
                int[] rank = [0, 0, 0];
                for (int i = 0; i < rank1.Length; i++)
                {
                    rank[i] = rank1[i];
                }

                return new HandRank(HandRank.HandRankType.OnePair, a.rank, rank[0], rank[1], rank[2]);
            }
        }

        #endregion

        #region High card

        {
            var rank1 = hand
                .Cards
                .Select(t => t.Rank)
                .OrderByDescending(t => t)
                .Take(5)
                .ToArray();
            int[] rank = [0, 0, 0, 0, 0];
            for (int i = 0; i < rank1.Length; i++)
            {
                rank[i] = rank1[i];
            }

            return new HandRank(HandRank.HandRankType.HighCard, rank[0], rank[1], rank[2], rank[3], rank[4]);
        }

        #endregion
    }

    #endregion
}
