﻿namespace PokerAIRL_Sharp;

/// <url>https://en.wikipedia.org/wiki/List_of_poker_hands</url>
public class HandRank : IComparable
{
    public HandRankType RankType { get; }
    public int SubType1 { get; }
    public int SubType2 { get; }
    public int SubType3 { get; }
    public int SubType4 { get; }
    public int SubType5 { get; }

    public static readonly HandRank Empty = new HandRank(HandRankType.UNKNOWN, 0);

    public HandRank(
        HandRankType rankType,
        int subType1,
        int subType2 = 0,
        int subType3 = 0,
        int subType4 = 0,
        int subType5 = 0
    )
    {
        RankType = rankType;
        SubType1 = subType1;
        SubType2 = subType2;
        SubType3 = subType3;
        SubType4 = subType4;
        SubType5 = subType5;
    }

    public int CompareTo(HandRank other)
    {
        if ((int)this.RankType != (int)other.RankType)
        {
            return this.RankType.CompareTo(other.RankType);
        }

        int arsCount = 0;

        switch (this.RankType)
        {
            case HandRankType.StraightFlush:
            case HandRankType.Straight:
                return this.SubType1.CompareTo(other.SubType1);
            case HandRankType.FourOfAKind:
            {
                var a = this.SubType1.CompareTo(other.SubType1);
                return a != 0 ? a : this.SubType2.CompareTo(other.SubType2);
            }
            case HandRankType.ThreeOfAKind:
            case HandRankType.TwoPair:
            case HandRankType.FullHouse:
            {
                arsCount = 3;
                break;
            }
            case HandRankType.Flush:
            case HandRankType.OnePair:
            case HandRankType.HighCard:
            {
                arsCount = 5;
                break;
            }
            default:
                throw new ArgumentOutOfRangeException();
        }

        var ars = new[]
        {
            (this.SubType1, other.SubType1),
            (this.SubType2, other.SubType2),
            (this.SubType3, other.SubType3),
            (this.SubType4, other.SubType4),
            (this.SubType5, other.SubType5),
        };

        foreach (var (a, b) in ars.Take(arsCount))
        {
            if (a == b) continue;
            return a.CompareTo(b);
        }

        return 0;
    }

    public int CompareTo(object? target)
    {
        return target switch
        {
            null => 1,
            HandRank handRank => CompareTo(handRank),
            _ => throw new ArgumentException("Object is not a HandRank", nameof(target))
        };
    }

    public override string ToString()
    {
        // ReSharper disable once ConvertSwitchStatementToSwitchExpression
        switch (this.RankType)
        {
            case HandRankType.StraightFlush:
            case HandRankType.Straight:
                return $"{RankType}: {SubType1}";
            case HandRankType.FourOfAKind:
                return $"{RankType}: {SubType1} {SubType2}";
            case HandRankType.FullHouse:
            case HandRankType.ThreeOfAKind:
            case HandRankType.TwoPair:
                return $"{RankType}: {SubType1} {SubType2} {SubType3}";
            case HandRankType.Flush:
            case HandRankType.OnePair:
            case HandRankType.HighCard:
                return $"{RankType}: {SubType1} {SubType2} {SubType3} {SubType4} {SubType5}";
            default: throw new ArgumentOutOfRangeException();
        }
    }

    public enum HandRankType
    {
        StraightFlush = 9,
        FourOfAKind = 8,
        FullHouse = 7,
        Flush = 6,
        Straight = 5,
        ThreeOfAKind = 4,
        TwoPair = 3,
        OnePair = 2,
        HighCard = 1,
        UNKNOWN = 0,
    }
}
