﻿using System.Collections.Concurrent;
using System.Diagnostics;
using System.Runtime.CompilerServices;

namespace PokerAIRL_Sharp;

internal static class Program
{
    private static async Task Main()
    {
        await DiveCommunity();
    }

    private static async Task DiveCommunity()
    {
        var cancellationTokenSource = new CancellationTokenSource();
        var stoppingToken = cancellationTokenSource.Token;

        {
            for (var card0Id = 12; card0Id >= 0; card0Id--)
            {
                var card0 = Card.FromNumber(card0Id);

                for (var card1Id = 25; card1Id >= 0; card1Id--)
                {
                    if (card0Id == card1Id) continue;
                    var card1 = Card.FromNumber(card1Id);
                    if (card1.Rank > card0.Rank)
                    {
                        continue;
                    }

                    var onlyHandCards = new Card[] { card0, card1 };
                    var onlyMyHand = new Hand(onlyHandCards);
                    Console.WriteLine("{0}; {1}\n", card0, card1);

                    var winCounts = Enumerable.Repeat(0L, 10).ToArray();
                    var drawCounts = winCounts.ToArray();
                    var usedRivalCounts = new HashSet<int>();
                    var cyclesDone = 0L;

                    const int workerCount = 9;
                    const int communityCount = 3;
                    const int jCount = 1000;

                    var sw1 = Stopwatch.StartNew();
                    var sw2 = new Stopwatch();
                    for (var metaI = 0; metaI < 1000; metaI++)
                    {
                        var community1 = CreateFullHand(onlyMyHand, 2 + communityCount);
                        var order = new WorkerOrder(onlyMyHand, community1, jCount);

                        sw2.Start();
                        var workers = Enumerable
                            .Range(0, workerCount)
                            .Select(_ => StartWorkerAsync(order, stoppingToken))
                            .ToArray();

                        await Task.WhenAll(workers);
                        sw2.Stop();

                        while (order.Result.TryDequeue(out var result))
                        {
                            cyclesDone += result.cycleDone;
                            for (int rivalCount = 0; rivalCount <= 9; rivalCount++)
                            {
                                var winCount = result.winsByRivals[rivalCount];
                                if (winCount == -1)
                                {
                                    continue;
                                }

                                usedRivalCounts.Add(rivalCount);
                                var drawCount = result.drawByRivals[rivalCount];
                                winCounts[rivalCount] += winCount;
                                drawCounts[rivalCount] += drawCount;
                            }
                        }
                    }

                    //*
                    sw1.Stop();
                    Console.WriteLine("Cycles done: {0}; {1} ({2:P3})",
                        cyclesDone, sw1.Elapsed, sw2.Elapsed / sw1.Elapsed);
                    foreach (var rivalCount in new int[] { 1, 2, 5, 7, 9 })
                    {
                        if (!usedRivalCounts.Contains(rivalCount)) continue;

                        Console.WriteLine(
                            "{0,1}: Win rate: {1,8:P2} & {2,9:P4}",
                            rivalCount,
                            (winCounts[rivalCount] * 1d) / cyclesDone,
                            (drawCounts[rivalCount] * 1d) / cyclesDone
                        );
                    }
                    // */

                    Console.WriteLine("====================");
                }
            }
        }
    }

    #region CreateFullHand

    private static Card[] CreateFullHand(int count)
    {
        return CreateFullHand([], count);
    }

    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    private static Card[] CreateFullHand(Hand hand, int count)
    {
        return CreateFullHand(hand.Cards, count);
    }

    private static Card[] CreateFullHand(IReadOnlyList<Card> origin, int count)
    {
        if (origin.Count >= count) return [];

        var cards = origin.ToHashSet();

        var rnd = new Random();
        do
        {
            var n = rnd.Next(0, 52);
            var card = Card.FromNumber(n);
            var u = cards.Contains(card);
            if (!u)
            {
                cards.Add(card);
            }
        } while (cards.Count < count);

        return cards.Skip(origin.Count).ToArray();
    }

    #endregion

    #region Workers

    public class WorkerOrder
    {
        public readonly Hand MyHand;
        public readonly Card[] Community;
        public readonly int JCount;

        public volatile int CycleDone = 0;

        public readonly ConcurrentQueue<(long cycleDone, long[] winsByRivals, long[] drawByRivals)> Result = new();

        public WorkerOrder(Hand myHand, Card[] community, int jCount)
        {
            MyHand = myHand;
            Community = community;
            JCount = jCount;
        }
    }

    public static Task StartWorkerAsync(WorkerOrder order, CancellationToken stoppingToken)
    {
        return Task.Run(() => StartWorker(order, stoppingToken), CancellationToken.None);
    }

    public static void StartWorker(WorkerOrder order, CancellationToken stoppingToken)
    {
        var onlyMyHand = order.MyHand;
        var communityCards = order.Community;

        var winCounts = Enumerable.Repeat(0L, 10).ToArray();
        var drawCounts = winCounts.ToArray();

        var results = order.Result;
        var cycleDone = 0L;

        while (true)
        {
            var number = Interlocked.Increment(ref order.CycleDone);
            cycleDone++;

            foreach (var rivalCount in new int[] { 1, 2, 5, 7, 9 })
            {
                var myWholeHand = onlyMyHand + communityCards;
                var communityAndRival = CreateFullHand(myWholeHand, 7 + rivalCount * 2);
                myWholeHand += communityAndRival.SkipLast(rivalCount * 2);
                var wholeCommunityCards = communityCards
                    .Concat(communityAndRival.SkipLast(rivalCount * 2))
                    .ToArray();

                var allHands = new List<(Hand hand, int rivalIndex)>(capacity: rivalCount + 1)
                {
                    (myWholeHand, -1),
                };
                for (int n = 0; n < rivalCount; n++)
                {
                    var rivalHand = new Hand(communityAndRival
                        .TakeLast(rivalCount * 2)
                        .Skip(n * 2)
                        .Take(2)) + wholeCommunityCards;
                    allHands.Add((rivalHand, n));
                }

                var a = allHands
                    .OrderByDescending(t => t.hand.HandRank)
                    .Take(2)
                    .ToArray();
                if (a[0].hand.HandRank == a[1].hand.HandRank)
                {
                    drawCounts[rivalCount]++;
                    continue;
                }

                var winner = a[0].rivalIndex;
                if (winner == -1)
                {
                    winCounts[rivalCount]++;
                }
            }

            if (number >= order.JCount)
            {
                break;
            }
        }

        /*
        var r = Enumerable.Repeat(-1L, 10 * 2).ToArray();
        var rivalCounts = winCounts
            .Select((t, index) => (t, index))
            .Concat(drawCounts.Select((t, index) => (t, index)))
            .Where(x => x.t > 0)
            .Select(t => t.index)
            .Distinct()
            .OrderBy(t => t)
            .ToArray();
        foreach (var rivalCount in rivalCounts)
        {
            r[rivalCount * 2] = winCounts[rivalCount];
            r[rivalCount * 2 + 1] = drawCounts[rivalCount];
        }
        // */

        results.Enqueue((cycleDone, winCounts, drawCounts));
    }

    #endregion
}
